/**
 * Bolt Website Scraper - Settings Manager
 * Handles user settings, auto-save features, and custom export options
 */

// Default settings
const DEFAULT_SETTINGS = {
    // Auto-save features
    autoSave: {
        enabled: false,
        interval: 5, // minutes
        format: 'csv', // csv or excel
        lastAutoSave: null
    },
    
    // Auto-download on completion
    autoDownload: {
        enabled: false,
        format: 'csv' // csv or excel
    },
    
    // Custom export settings
    customExport: {
        columns: {
            url: true,
            pageTitle: true,
            email1: true,
            email2: true,
            phone1: true,
            phone2: true,
            aboutPageUrl: true,
            contactPageUrl: true,
            facebookLink: true,
            instagramLinks: true,
            linkedinLinks: true,
            tiktokLinks: true,
            youtubeLinks: true,
            whatsappLinks: true,
            telegramLinks: true,
            pinterestLinks: true,
            twitterLinks: true,
            skypeLinks: true,
            snapchatLinks: true
        }
    },
    
    // UI settings
    ui: {
        darkMode: true,
        tableRowsPerPage: 10
    }
};

class SettingsManager {
    constructor() {
        this.settings = null;
        this.autoSaveTimer = null;
    }
    
    /**
     * Initialize settings
     */
    async init() {
        await this.loadSettings();
        this.setupAutoSave();
        return this.settings;
    }
    
    /**
     * Load settings from storage
     */
    async loadSettings() {
        try {
            const result = await chrome.storage.local.get('boltScraperSettings');
            this.settings = result.boltScraperSettings || DEFAULT_SETTINGS;
        } catch (error) {
            console.error('Error loading settings:', error);
            this.settings = DEFAULT_SETTINGS;
        }
        return this.settings;
    }
    
    /**
     * Save settings to storage
     */
    async saveSettings() {
        try {
            await chrome.storage.local.set({ 'boltScraperSettings': this.settings });
            return true;
        } catch (error) {
            console.error('Error saving settings:', error);
            return false;
        }
    }
    
    /**
     * Update a specific setting
     * @param {string} path - Dot notation path to the setting (e.g., 'autoSave.enabled')
     * @param {any} value - New value for the setting
     */
    async updateSetting(path, value) {
        const parts = path.split('.');
        let current = this.settings;
        
        // Navigate to the nested property
        for (let i = 0; i < parts.length - 1; i++) {
            if (!current[parts[i]]) {
                current[parts[i]] = {};
            }
            current = current[parts[i]];
        }
        
        // Set the value
        current[parts[parts.length - 1]] = value;
        
        // If updating auto-save settings, restart the timer
        if (path.startsWith('autoSave.')) {
            this.setupAutoSave();
        }
        
        return await this.saveSettings();
    }
    
    /**
     * Get a specific setting
     * @param {string} path - Dot notation path to the setting
     * @param {any} defaultValue - Default value if setting doesn't exist
     */
    getSetting(path, defaultValue = null) {
        const parts = path.split('.');
        let current = this.settings;
        
        for (const part of parts) {
            if (current === undefined || current === null || !current.hasOwnProperty(part)) {
                return defaultValue;
            }
            current = current[part];
        }
        
        return current;
    }
    
    /**
     * Setup auto-save timer
     */
    setupAutoSave() {
        // Clear existing timer
        if (this.autoSaveTimer) {
            clearInterval(this.autoSaveTimer);
            this.autoSaveTimer = null;
        }
        
        // If auto-save is enabled, set up the timer
        if (this.settings.autoSave.enabled) {
            const intervalMs = this.settings.autoSave.interval * 60 * 1000;
            this.autoSaveTimer = setInterval(() => {
                this.performAutoSave();
            }, intervalMs);
        }
    }
    
    /**
     * Perform auto-save operation
     */
    async performAutoSave() {
        // This function will be called by the timer
        // The actual implementation will be in popup.js
        chrome.runtime.sendMessage({ action: 'perform_auto_save' });
        
        // Update last auto-save timestamp
        this.settings.autoSave.lastAutoSave = new Date().toISOString();
        await this.saveSettings();
    }
    
    /**
     * Get custom export data based on selected columns
     * @param {Array} data - Full data array
     */
    getCustomExportData(data) {
        const selectedColumns = this.settings.customExport.columns;
        
        // Filter the data to include only selected columns
        return data.map(row => {
            const filteredRow = {};
            for (const key in row) {
                if (selectedColumns[key]) {
                    filteredRow[key] = row[key];
                }
            }
            return filteredRow;
        });
    }
    
    /**
     * Reset settings to defaults
     */
    async resetSettings() {
        this.settings = JSON.parse(JSON.stringify(DEFAULT_SETTINGS));
        this.setupAutoSave();
        return await this.saveSettings();
    }
}

// Create and export a singleton instance
const settingsManager = new SettingsManager();

// Initialize settings when the script loads
settingsManager.init().catch(error => console.error('Failed to initialize settings:', error));

// Make it accessible globally
window.settingsManager = settingsManager; 